
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <stdio.h>

#include "interface.h"
#include "vphysics_interface.h"
#include "eiface.h"
#include "cbase.h"
#include "gmodinterface/gmluamodule.h"
#include "vphysics/performance.h"
#include "vcollide_parse.h"
#include "trace.h"

#include "vector.h"

#include "tier0/memdbgon.h"

// defines
#define TYPE_PHYSENV 5403


// interfaces
IVEngineServer* engine = NULL;
IPhysics* physics = NULL;
IPhysicsCollision* physcollision = NULL;
IPhysicsSurfaceProps* physprops = NULL;
IVModelInfo* modelinfo = NULL;
IServerGameEnts* servergameents = NULL;

// main physics environment
IPhysicsEnvironment* physenv = NULL;


//------ Setup module
GMOD_MODULE( Init, Shutdown );

/*-------------------------------------
	g_PhysDefaultObjectParams
	Port from the SDK
-------------------------------------*/
const objectparams_t g_PhysDefaultObjectParams =
{
	NULL,
	1.0, //mass
	1.0, // inertia
	0.1f, // damping
	0.1f, // rotdamping
	0.05f, // rotIntertiaLimit
	"DEFAULT",
	NULL, // game data
	0.f, // volume (leave 0 if you don't have one or call physcollision->CollideVolume() to compute it)
	1.0f, // drag coefficient
	true, // enable collisions?
};


/*-------------------------------------
	CSolidSetDefaults
	Port from the SDK
-------------------------------------*/
class CSolidSetDefaults : public IVPhysicsKeyHandler
{

public:

	virtual void ParseKeyValue( void *pData, const char *pKey, const char *pValue );
	virtual void SetDefaults( void *pData );

	unsigned int GetContentsMask() { return m_contentsMask; }

private:

	unsigned int m_contentsMask;

};
void CSolidSetDefaults::ParseKeyValue( void *pData, const char *pKey, const char *pValue )
{

	if ( !Q_stricmp( pKey, "contents" ) )
	{

		m_contentsMask = atoi( pValue );

	}

}
void CSolidSetDefaults::SetDefaults( void *pData )
{

	solid_t *pSolid = (solid_t *)pData;
	pSolid->params = g_PhysDefaultObjectParams;
	m_contentsMask = CONTENTS_SOLID;

}
CSolidSetDefaults g_SolidSetup;




/*-------------------------------------
	PhysModelParseSolidByIndex
	Port from the SDK
-------------------------------------*/
bool PhysModelParseSolidByIndex( solid_t &solid, CBaseEntity *pEntity, int modelIndex, int solidIndex )
{

	vcollide_t *pCollide = modelinfo->GetVCollide( modelIndex );
	if ( !pCollide )
		return false;

	bool parsed = false;

	memset( &solid, 0, sizeof(solid) );
	solid.params = g_PhysDefaultObjectParams;

	IVPhysicsKeyParser *pParse = physcollision->VPhysicsKeyParserCreate( pCollide->pKeyValues );
	while ( !pParse->Finished() )
	{
		const char *pBlock = pParse->GetCurrentBlockName();
		if ( !strcmpi( pBlock, "solid" ) )
		{
			solid_t tmpSolid;
			memset( &tmpSolid, 0, sizeof(tmpSolid) );
			tmpSolid.params = g_PhysDefaultObjectParams;

			pParse->ParseSolid( &tmpSolid, &g_SolidSetup );

			if ( solidIndex < 0 || tmpSolid.index == solidIndex )
			{
				parsed = true;
				solid = tmpSolid;
				// just to be sure we aren't ever getting a non-zero solid by accident
				Assert( solidIndex >= 0 || solid.index == 0 );
				break;
			}
		}
		else
		{
			pParse->SkipBlock();
		}
	}
	physcollision->VPhysicsKeyParserDestroy( pParse );

	// collisions are off by default
	solid.params.enableCollisions = true;

	solid.params.pGameData = static_cast<void *>(pEntity);
	solid.params.pName = STRING(pEntity->GetModelName());
	return parsed;

}

/*-------------------------------------
	FindFieldByName
-------------------------------------*/
typedescription_t *FindFieldByName( const char *fieldname, datamap_t *dmap )
{

	// number of fields
	int c = dmap->dataNumFields;

	// find the field we want
	for( int i = 0; i < c; i++ )
	{

		// get the type description for this field
		typedescription_t* td = &dmap->dataDesc[i];

		// ignore void fields
		if( td->fieldType == FIELD_VOID )
		{

			continue;

		}

		// Embedded, do a recursive search
		if( td->fieldType == FIELD_EMBEDDED )
		{

			typedescription_t* ret = FindFieldByName( fieldname, td->td );
			
			// found?
			if( ret )
			{

				return ret;

			}

		}

		// matching name?
		if( stricmp( td->fieldName, fieldname ) == 0 )
		{

			return td;

		}

	}

        // base map eh? assuming it's base class's map
	if( dmap->baseMap )
	{
		
		return FindFieldByName( fieldname, dmap->baseMap );

	}

	return NULL;

}

/*-------------------------------------
	FindOffsetDataDescMap
-------------------------------------*/
int FindOffsetDataDescMap( datamap_t *dmap, const char *fieldname )
{

	typedescription_t* td = FindFieldByName( fieldname, dmap );

	// valid?
	if( td )
	{

		return td->fieldOffset[0];

	}

	return 0;

}

/*-------------------------------------
	GetGravity
-------------------------------------*/
LUA_FUNCTION( GetGravity )
{

	// get the gravity
	Vector gravity;
	physenv->GetGravity( &gravity );

	// push
	PushVector( L, gravity );

	return 1;

}


/*-------------------------------------
	SetGravity
-------------------------------------*/
LUA_FUNCTION( SetGravity )
{

	// check arguments
	Lua()->CheckType( 1, GLua::TYPE_VECTOR );

	// set
	physenv->SetGravity( *GetVector( L, 1 ) );

	return 0;

}


/*-------------------------------------
	SetAirDensity
-------------------------------------*/
LUA_FUNCTION( SetAirDensity )
{

	// check arguments
	Lua()->CheckType( 1, GLua::TYPE_NUMBER );

	// set
	physenv->SetAirDensity( Lua()->GetNumber( 1 ) );

	return 0;

}

/*-------------------------------------
	GetAirDensity
-------------------------------------*/
LUA_FUNCTION( GetAirDensity )
{

	// push
	Lua()->Push( physenv->GetAirDensity() );

	return 1;

}


/*-------------------------------------
	SetPerformanceSettings
-------------------------------------*/
LUA_FUNCTION( SetPerformanceSettings )
{

	// check arguments
	Lua()->CheckType( 1, GLua::TYPE_TABLE );

	// fetch
	ILuaObject* table = Lua()->GetObject( 1 );

	// setup a settings
	physics_performanceparams_t settings;
		settings.Defaults();

	// get settings
	ILuaObject* maxCollisionsPerObjectPerTimestep = table->GetMember( "maxCollisionsPerObjectPerTimestep" );
	ILuaObject* maxCollisionChecksPerTimestep = table->GetMember( "maxCollisionChecksPerTimestep" );
	ILuaObject* maxVelocity = table->GetMember( "maxVelocity" );
	ILuaObject* maxAngularVelocity = table->GetMember( "maxAngularVelocity" );
	ILuaObject* lookAheadTimeObjectsVsWorld = table->GetMember( "lookAheadTimeObjectsVsWorld" );
	ILuaObject* lookAheadTimeObjectsVsObject = table->GetMember( "lookAheadTimeObjectsVsObject" );
	ILuaObject* minFrictionMass = table->GetMember( "minFrictionMass" );
	ILuaObject* maxFrictionMass = table->GetMember( "maxFrictionMass" );

	// maxCollisionsPerObjectPerTimestep
	if( !maxCollisionsPerObjectPerTimestep->isNil() )
	{

		settings.maxCollisionsPerObjectPerTimestep = maxCollisionsPerObjectPerTimestep->GetInt();

	}

	// maxCollisionChecksPerTimestep
	if( !maxCollisionChecksPerTimestep->isNil() )
	{

		settings.maxCollisionChecksPerTimestep = maxCollisionChecksPerTimestep->GetInt();

	}

	// maxVelocity
	if( !maxVelocity->isNil() )
	{

		settings.maxVelocity = maxVelocity->GetFloat();

	}

	// maxAngularVelocity
	if( !maxAngularVelocity->isNil() )
	{

		settings.maxAngularVelocity = maxAngularVelocity->GetFloat();

	}

	// lookAheadTimeObjectsVsWorld
	if( !lookAheadTimeObjectsVsWorld->isNil() )
	{

		settings.lookAheadTimeObjectsVsWorld = lookAheadTimeObjectsVsWorld->GetFloat();

	}

	// lookAheadTimeObjectsVsObject
	if( !lookAheadTimeObjectsVsObject->isNil() )
	{

		settings.lookAheadTimeObjectsVsObject = lookAheadTimeObjectsVsObject->GetFloat();

	}

	// minFrictionMass
	if( !minFrictionMass->isNil() )
	{

		settings.minFrictionMass = minFrictionMass->GetFloat();

	}

	// maxFrictionMass
	if( !maxFrictionMass->isNil() )
	{

		settings.maxFrictionMass = maxFrictionMass->GetFloat();

	}

	// set
	physenv->SetPerformanceSettings( &settings );

	return 0;

}


/*-------------------------------------
	GetPerformanceSettings
-------------------------------------*/
LUA_FUNCTION( GetPerformanceSettings )
{

	// setup a settings
	physics_performanceparams_t settings;
	physenv->GetPerformanceSettings( &settings );

	// construct table
	ILuaObject* results = Lua()->GetNewTable();
	results->SetMember( "maxCollisionsPerObjectPerTimestep", (float)settings.maxCollisionsPerObjectPerTimestep );
	results->SetMember( "maxCollisionChecksPerTimestep", (float)settings.maxCollisionChecksPerTimestep );
	results->SetMember( "maxVelocity", settings.maxVelocity );
	results->SetMember( "maxAngularVelocity", settings.maxAngularVelocity );
	results->SetMember( "lookAheadTimeObjectsVsWorld", settings.lookAheadTimeObjectsVsWorld );
	results->SetMember( "lookAheadTimeObjectsVsObject", settings.lookAheadTimeObjectsVsObject );
	results->SetMember( "minFrictionMass", settings.minFrictionMass );
	results->SetMember( "maxFrictionMass", settings.maxFrictionMass );

	// push
	results->Push();

	return 1;

}

/*-------------------------------------
	SetSimulationTimestep
-------------------------------------*/
LUA_FUNCTION( SetSimulationTimestep )
{

	// check arguments
	Lua()->CheckType( 1, GLua::TYPE_NUMBER );

	// set
	physenv->SetSimulationTimestep( Lua()->GetNumber( 1 ) );

	return 0;

}

/*-------------------------------------
	GetSimulationTimestep
-------------------------------------*/
LUA_FUNCTION( GetSimulationTimestep )
{

	// push
	Lua()->Push( physenv->GetSimulationTimestep() );

	return 1;

}

/*-------------------------------------
	ScalePhysicsObject
-------------------------------------*/
LUA_FUNCTION( ScalePhysicsObject )
{

	Lua()->CheckType( 1, GLua::TYPE_PHYSOBJ );
	Lua()->CheckType( 2, GLua::TYPE_VECTOR );

	// fetch the physobj, scale, and get it's entity
	IPhysicsObject* object = *(IPhysicsObject**)Lua()->GetUserData( 1 );
	Vector& scale = *GetVector( L, 2 );
	CBaseEntity* entity = static_cast<CBaseEntity*>( object->GetGameData() );

	// create a collision query
	CPhysCollide* collide = const_cast<CPhysCollide*>( object->GetCollide() );
	ICollisionQuery* query = physcollision->CreateQueryModel( collide );

	// get the convex hulls of the object
	int hull_count = query->ConvexCount();
	CPhysConvex** hulls = new CPhysConvex*[ hull_count ];

	// query for triangles of each
	for( int i = 0; i < hull_count; i++ )
	{

		// get the triangle verts
		int triangle_count = query->TriangleCount( i );
		Vector** vertices = new Vector*[ triangle_count ];
		for( int j = 0; j < triangle_count; j++ )
		{

			vertices[ j ] = new Vector[ 3 ];

			// fetch triangle
			query->GetTriangleVerts( i, j, vertices[ j ] );

			// scale triangles
			vertices[ j ][ 0 ] *= scale;
			vertices[ j ][ 1 ] *= scale;
			vertices[ j ][ 2 ] *= scale;

		}

		// create a new convex hull from these triangle verts
		hulls[ i ] = physcollision->ConvexFromVerts( vertices, triangle_count );

		// cleanup our mess
		for( int j = 0; j < triangle_count; j++ )
		{

			delete [] vertices[ j ];

		}
		delete [] vertices;

	}

	// no longer need the query model
	physcollision->DestroyQueryModel( query );

	// get a new collide object based on our hull collection
	collide = physcollision->ConvertConvexToCollide( hulls, hull_count );

	// cleanup
	delete [] hulls;

	// parse the old solid.
	solid_t solid;
	PhysModelParseSolidByIndex( solid, entity, entity->GetModelIndex(), -1 );

	// scale the solid attributes, linear.
	float fScale = ( scale.x + scale.y + scale.z ) * 0.33;
	solid.params.mass *= fScale;
	solid.params.volume *= fScale;
	solid.params.inertia *= fScale;

	// get original position
	Vector position;
	QAngle angles;
	object->GetPosition( &position, &angles );

	// create a new poly object
	IPhysicsObject* scaled_object = physenv->CreatePolyObject( collide, object->GetMaterialIndex(), position, angles, &solid.params );
	scaled_object->SetGameData( entity );

	// find the offset for physics object
	static int PhysObjectOffset = 0;
	if( !PhysObjectOffset )
	{

		// fetch datamap
		datamap_t* map = entity->GetDataDescMap();

		// find the m_pPhysicsObject field
		PhysObjectOffset = FindOffsetDataDescMap( map, "m_pPhysicsObject" );

	}

	// set
	if( PhysObjectOffset )
	{

		// replace
		IPhysicsObject* oldphysobj = *(IPhysicsObject**)( (char*)entity + PhysObjectOffset );
	
		// valid?
		if( oldphysobj )
		{

			// spew a nice warning
			Warning( "Overriding physics object!\n" );

			// destroy it
			physenv->DestroyObject( oldphysobj );

		}

		// set
		*(IPhysicsObject**)( (char*)entity + PhysObjectOffset ) = scaled_object;

	}
	else
	{

		// destroy the physics object
		physenv->DestroyObject( scaled_object );

		// error
		Lua()->Error( "Unable to find the offset for m_pPhysicsObject, not setting physics object." );

	}

	return 0;

}


/*-------------------------------------
	CreateEnvironment
-------------------------------------*/
LUA_FUNCTION( CreateEnvironment )
{

	// create
	IPhysicsEnvironment* env = physics->CreateEnvironment();

	// push
	ILuaObject* PhysEnv = Lua()->GetMetaTable( "PhysEnv", TYPE_PHYSENV );
	Lua()->PushUserData( PhysEnv, env );

	return 1;

}

/*-------------------------------------
	DestroyEnvironment
-------------------------------------*/
LUA_FUNCTION( DestroyEnvironment )
{

	Lua()->CheckType( 1, TYPE_PHYSENV );

	// create
	IPhysicsEnvironment* env = (IPhysicsEnvironment*)Lua()->GetUserData( 1 );
	if( env )
	{

		physics->DestroyEnvironment( env );

	}

	return 0;

}

/*-------------------------------------
	TransferObject
-------------------------------------*/
LUA_FUNCTION( TransferObject )
{

	Lua()->CheckType( 1, GLua::TYPE_PHYSOBJ );
	Lua()->CheckType( 2, TYPE_PHYSENV );

	// physobj
	IPhysicsObject* physobj = *(IPhysicsObject**)Lua()->GetUserData( 1 );

	// environemnt
	IPhysicsEnvironment* env = (IPhysicsEnvironment*)Lua()->GetUserData( 2 );

	// set
	//if( env && physobj )
	//{

		physenv->TransferObject( physobj, env );

	//}

	return 0;

}

/*-------------------------------------
	Init
-------------------------------------*/
int Init( lua_State* L )
{

	//--- Engine
	CreateInterfaceFn EngineFactory = Sys_GetFactory( "engine.dll" );

	// get the server interface
	engine = (IVEngineServer*)EngineFactory( INTERFACEVERSION_VENGINESERVER, NULL );
	modelinfo = (IVModelInfo*)EngineFactory( VMODELINFO_SERVER_INTERFACE_VERSION, NULL );
	if( !engine || !modelinfo )
	{

		// not critical
		Lua()->Error( "Failed to get required engine interfaces.\n" );

	}


	//--- Game
	CreateInterfaceFn GameFactory = Sys_GetFactory( "server.dll" );
	servergameents = (IServerGameEnts*)GameFactory( INTERFACEVERSION_SERVERGAMEENTS, NULL );
	if( !servergameents )
	{

		// not critical
		Lua()->Error( "Failed to get required game server interfaces.\n" );

	}


	//----- Physics
	CreateInterfaceFn PhysicsFactory = Sys_GetFactory( "vphysics.dll" );

	// get physics interface
	physics = (IPhysics*)PhysicsFactory( VPHYSICS_INTERFACE_VERSION, NULL );
	physprops = (IPhysicsSurfaceProps*)PhysicsFactory( VPHYSICS_SURFACEPROPS_INTERFACE_VERSION, NULL );
	physcollision = (IPhysicsCollision*)PhysicsFactory( VPHYSICS_COLLISION_INTERFACE_VERSION, NULL );
	if( !physics || !physcollision || !physprops )
	{

		Lua()->Error( "Failed to get required physics interfaces.\n" );

	}

	// get physenv and reverse gravity
	physenv = physics->GetActiveEnvironmentByIndex( 0 );
	if( !physenv )
	{

		Lua()->Error( "Failed to get physics environment.\n" );

	}

	// create the physenv table
	Lua()->NewGlobalTable( "physics" );

	// get the physenv table
	ILuaObject* phys = Lua()->GetGlobal( "physics" );
	if( phys )
	{

		// physics methods
		phys->SetMember( "GetAirDensity", GetAirDensity );
		phys->SetMember( "SetAirDensity", SetAirDensity );
		phys->SetMember( "GetGravity", GetGravity );
		phys->SetMember( "SetGravity", SetGravity );
		phys->SetMember( "SetPerformanceSettings", SetPerformanceSettings );
		phys->SetMember( "GetPerformanceSettings", GetPerformanceSettings );
		phys->SetMember( "GetSimulationTimestep", GetSimulationTimestep );
		phys->SetMember( "SetSimulationTimestep", SetSimulationTimestep );
		phys->SetMember( "CreateEnvironment", CreateEnvironment );
		//phys->SetMember( "DestroyEnvironment", DestroyEnvironment );

	}


	// Extend PhysObj
	ILuaObject* PhysObj = Lua()->GetMetaTable( "PhysObj", GLua::TYPE_PHYSOBJ );
	if( PhysObj )
	{

		// get __index
		ILuaObject* PhysObjMethods = PhysObj->GetMember( "__index" );
		if( PhysObjMethods )
		{

			PhysObjMethods->SetMember( "Scale", ScalePhysicsObject );
			PhysObjMethods->SetMember( "TransferObject", TransferObject );

		}

	}

	// phys environment
	ILuaObject* PhysEnv = Lua()->GetMetaTable( "PhysEnv", TYPE_PHYSENV );
	if( PhysEnv )
	{

		// __gc
		PhysEnv->SetMember( "__gc", DestroyEnvironment );

	}


	return 0;

}


/*-------------------------------------
	Shutdown
-------------------------------------*/
int Shutdown( lua_State* L )
{

	return 0;

}
