
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <winsock2.h>

#include <stdlib.h>
#include <stdio.h>
#include "common/GMLuaModule.h"

#include <mysql.h>

GMOD_MODULE( Init, Quit );

#define GM_MYSQL_QUERY_NUMERIC		0x01
#define GM_MYSQL_QUERY_FIELDS		0x02
#define GM_MYSQL_QUERY_NOBLOCK		0x04
#define GM_MYSQL_QUERY_BOTH			(GM_MYSQL_QUERY_NUMERIC | GM_MYSQL_QUERY_FIELDS)
#define GM_MYSQL_QUERY_FLAG_MASK	(GM_MYSQL_QUERY_BOTH | GM_MYSQL_QUERY_NOBLOCK)

typedef struct
{
	MYSQL* pSQL;
	char sQuery[1024];
	int nResultType;

	int nNumRows;
	int nNumColumns;
	char** szColumnHeadings;
	char*** szRows;

	bool bComplete;
	bool bAbort;
	bool bError;
	bool bUsed;
	char sErrorMessage[1024];

#ifdef WIN32
	HANDLE hThread;
	HANDLE hWaitObject;
	HANDLE hContinueObject;
	DWORD dwThreadId;
#else
	#error LINUX!
#endif
} mysql_thread_data;

#define MAX_MYSQL_CONNECTIONS	255
#define MAX_MYSQL_THREADS		1024
MYSQL* g_pMysqlInstances[ MAX_MYSQL_CONNECTIONS ] = {0};
mysql_thread_data g_MySqlThreads[ MAX_MYSQL_THREADS ] = {0};

#define S(x) #x

LUA_FUNCTION( connect )
{
	int nNumParameters = g_Lua->Top();

	if (nNumParameters < 3)
	{
		g_Lua->LuaError("Insufficient parameters!\n");
		return 0;
	}

	const char* sHost = g_Lua->GetString(1);
	const char* sUser = g_Lua->GetString(2);
	const char* sPass = g_Lua->GetString(3);
	const char* sDb = 0;
	unsigned int nPort = 0;
	const char* sUnixSkt = 0;
	unsigned int nClientFlag = 0;

	if (nNumParameters > 3)	sDb = g_Lua->GetString(4);
	if (nNumParameters > 4)	nPort = (unsigned int)g_Lua->GetInteger(5);
	if (nNumParameters > 5)	sUnixSkt = g_Lua->GetString(6);
	if (nNumParameters > 6)	nClientFlag = (unsigned int)g_Lua->GetInteger(7);

	MYSQL** ppFreeSlot = 0;
	unsigned int nFreeSlot = 0;

	for (unsigned int i=0; i<MAX_MYSQL_CONNECTIONS; i++)
	{
		if (g_pMysqlInstances[i] == 0)
		{
			ppFreeSlot = &g_pMysqlInstances[i];
			nFreeSlot = i;
			break;
		}
	}
	if (ppFreeSlot == 0)
	{
		g_Lua->Push( 0.0f );
		g_Lua->Push("Reached max connections - " S(MAX_MYSQL_CONNECTIONS) "!");
		return 2;
	}

	*ppFreeSlot = mysql_init(0);
	if (*ppFreeSlot == 0)
	{
		g_Lua->Push( 0.0f );
		g_Lua->Push("mysql_init failed - probably out of memory!");
		return 2;
	}

	if (!mysql_real_connect(*ppFreeSlot, sHost, sUser, sPass, sDb, nPort, sUnixSkt, nClientFlag))
	{
		g_Lua->Push( 0.0f );
		g_Lua->Push( mysql_error(*ppFreeSlot) );
		return 2;
	}

	g_Lua->Push( (float)(nFreeSlot+1) );
	g_Lua->Push( "Connected Ok" );
	return 2;
}

LUA_FUNCTION( disconnect )
{
	int nNumParameters = g_Lua->Top();

	if (nNumParameters < 1)
	{
		g_Lua->LuaError("Insufficient parameters!\n");
		return 0;
	}

	unsigned int n = (unsigned int)g_Lua->GetInteger(1);
	n--;

	if (n >= MAX_MYSQL_CONNECTIONS || g_pMysqlInstances[n] == 0)
	{
		g_Lua->Push(false);
		g_Lua->Push("Invalid connection!");
		return 2;
	}

	mysql_close( g_pMysqlInstances[n] );
	g_pMysqlInstances[n] = 0;

	g_Lua->Push(true);
	g_Lua->Push("Connection closed!");
	return 2;
}

LUA_FUNCTION( insert_id )
{
	int nNumParameters = g_Lua->Top();

	if (nNumParameters < 1)
	{
		g_Lua->LuaError("Insufficient parameters!\n");
		return 0;
	}

	unsigned int n = (unsigned int)g_Lua->GetInteger(1);
	n--;

	if (n >= MAX_MYSQL_CONNECTIONS || g_pMysqlInstances[n] == 0)
	{
		g_Lua->Push(false);
		g_Lua->Push("Invalid connection!");
		return 2;
	}

	g_Lua->Push( (float)mysql_insert_id(g_pMysqlInstances[n]) );
	return 1;
}

LUA_FUNCTION( escape )
{
	int nNumParameters = g_Lua->Top();

	if (nNumParameters < 2)
	{
		g_Lua->LuaError("Insufficient parameters!\n");
		return 0;
	}

	unsigned int n = (unsigned int)g_Lua->GetInteger(1);
	n--;

	if (n >= MAX_MYSQL_CONNECTIONS || g_pMysqlInstances[n] == 0)
	{
		g_Lua->Push(false);
		g_Lua->Push("Invalid connection!");
		return 2;
	}

	const char* sQuery = g_Lua->GetString(2);
	char* sEscapedQuery = (char*)malloc( (strlen(sQuery) * 2) + 4 );

	if (sEscapedQuery == 0)
	{
		g_Lua->Push(false);
		g_Lua->Push("Out of memory!");
		return 2;
	}

	MYSQL* pSQL = g_pMysqlInstances[n];

	mysql_real_escape_string(pSQL, sEscapedQuery, sQuery, (unsigned long)strlen(sQuery) );

	g_Lua->Push( sEscapedQuery );

	free(sEscapedQuery);

	return 1;
}

void DoQuery(mysql_thread_data* pData)
{
	pData->bError = pData->bComplete = false;
	pData->nNumColumns = pData->nNumRows = 0;
	pData->szColumnHeadings = 0;
	pData->szRows = 0;

	if (mysql_query(pData->pSQL, pData->sQuery) != 0)
	{
		pData->bError = true;
		pData->bComplete = true;
		_snprintf(pData->sErrorMessage, sizeof(pData->sErrorMessage), "%s", mysql_error(pData->pSQL) );
		return;
	}

	MYSQL_RES* pResult = mysql_use_result(pData->pSQL);

	if (pResult == 0)
	{
		if(mysql_errno(pData->pSQL))  // mysql_fetch_row() failed due to an error
		{
			pData->bError = true;
			_snprintf(pData->sErrorMessage, sizeof(pData->sErrorMessage), "%s", mysql_error(pData->pSQL) );
		}
		else
		{
			pData->bError = false;
			_snprintf(pData->sErrorMessage, sizeof(pData->sErrorMessage), "%s", "OK" );
		}
		pData->bComplete = true;
		return;
	}

	pData->nNumColumns = mysql_num_fields(pResult);
	pData->nNumRows = 0;
	MYSQL_ROW CurrentRow = mysql_fetch_row(pResult);
	if (pData->nResultType & GM_MYSQL_QUERY_FIELDS)
	{
		pData->szColumnHeadings = (char**)malloc( sizeof(char*) * pData->nNumColumns );

		if (pData->szColumnHeadings)
		{
			for(int i = 0; i < pData->nNumColumns; i++)
			{
				char* sName = mysql_fetch_field_direct(pResult, i)->name;

				pData->szColumnHeadings[i] = (char*)malloc( strlen(sName)+1 );
				if (pData->szColumnHeadings[i])
				{
					_snprintf(pData->szColumnHeadings[i], strlen(sName)+1, "%s", sName);
				}
			}
		}
	}
	while (CurrentRow)
	{
		pData->szRows = (char***)realloc(pData->szRows, sizeof(char**) * (pData->nNumRows+1));
		pData->szRows[pData->nNumRows] = (char**)malloc( sizeof(char*) * pData->nNumColumns );
		for(int i = 0; i < pData->nNumColumns; i++)
		{
			if (CurrentRow[i])
			{
				pData->szRows[pData->nNumRows][i] = (char*)malloc( strlen(CurrentRow[i]) + 1 );
				_snprintf(pData->szRows[pData->nNumRows][i], strlen(CurrentRow[i])+1, "%s", CurrentRow[i]);
			}
			else
			{
				pData->szRows[pData->nNumRows][i] = 0;
			}
		}
		CurrentRow = mysql_fetch_row(pResult);
		pData->nNumRows++;

		if (pData->bAbort)
			break;

#ifdef WIN32
		if (pData->hWaitObject && pData->hContinueObject)
		{
			SetEvent(pData->hWaitObject);
			WaitForSingleObject(pData->hContinueObject, INFINITE);
			ResetEvent(pData->hWaitObject);
		}
#else
		#error LINUX
#endif
	}

	if(mysql_errno(pData->pSQL))  // mysql_fetch_row() failed due to an error
	{
		pData->bError = true;
		_snprintf(pData->sErrorMessage, sizeof(pData->sErrorMessage), "%s", mysql_error(pData->pSQL) );
	}
	else
	{
		pData->bError = false;
		_snprintf(pData->sErrorMessage, sizeof(pData->sErrorMessage), "%s", "OK" );
	}

	mysql_free_result(pResult);

	pData->bComplete = true;
}

void WaitForMySqlThread(mysql_thread_data* pData, unsigned long dwTimeout)
{
#ifdef WIN32
	WaitForSingleObject(pData->hWaitObject, dwTimeout);
#else
	#error LINUX
#endif
}

void ContinueMySqlThread(mysql_thread_data* pData)
{
#ifdef WIN32
	ResetEvent(pData->hContinueObject);
#else
	#error LINUX
#endif
}

void PushMySqlData(mysql_thread_data* pData)
{
	ILuaObject* pTable = g_Lua->GetNewTable();

	if (pData->szRows)
	{
		for (int i=0; i<pData->nNumRows; i++)
		{
			ILuaObject* pSubTable = g_Lua->GetNewTable();
			for (int j=0; j<pData->nNumColumns; j++)
			{
				if (pData->szRows[i][j])
				{
					if (pData->nResultType & GM_MYSQL_QUERY_NUMERIC)	pSubTable->SetMember((float)(j+1), pData->szRows[i][j]);
					if (pData->nResultType & GM_MYSQL_QUERY_FIELDS)		pSubTable->SetMember( pData->szColumnHeadings[j], pData->szRows[i][j]);
				}
			}

			pTable->SetMember((float)(i+1), pSubTable);

			SAFE_UNREF( pSubTable );
		}
	}

	g_Lua->Push( pTable );
	g_Lua->Push( !pData->bError );
	g_Lua->Push( pData->sErrorMessage );

	SAFE_UNREF( pTable );
}

void FreeMySqlData(mysql_thread_data* pData)
{
	if (pData->szColumnHeadings)
	{
		for (int i=0; i<pData->nNumColumns; i++)
		{
			free(pData->szColumnHeadings[i]);
		}
		free(pData->szColumnHeadings);
	}
	if (pData->szRows)
	{
		for (int i=0; i<pData->nNumRows; i++)
		{
			for (int j=0; j<pData->nNumColumns; j++)
			{
				if (pData->szRows[i][j])
				{
					free(pData->szRows[i][j]);
				}
			}
			free(pData->szRows[i]);
		}
		free(pData->szRows);
	}

#ifdef WIN32
	if (pData->hContinueObject)	CloseHandle(pData->hContinueObject);
	if (pData->hWaitObject)		CloseHandle(pData->hWaitObject);
#endif

	memset(pData, 0, sizeof(mysql_thread_data));
}

void WaitForThreadDeath(mysql_thread_data* pData)
{
	while (true)
	{
		DWORD dwExit = 0;
		GetExitCodeThread(pData->hThread, &dwExit);
		if (dwExit != STILL_ACTIVE)
			break;

		::SleepEx( 1000, TRUE );
	}
}

LUA_FUNCTION( thread_query_cleanup )
{
	int nNumParameters = g_Lua->Top();
	if (nNumParameters < 1)
	{
		g_Lua->LuaError("Insufficient parameters!\n");
		return 0;
	}

	unsigned int n = (unsigned int)g_Lua->GetInteger(1);
	n--;

	if (n >= MAX_MYSQL_THREADS || !g_MySqlThreads[n].bUsed)
	{
		g_Lua->LuaError("Invalid thread!\n");
		return 0;
	}

	g_MySqlThreads[n].bAbort = true;
	WaitForThreadDeath(&g_MySqlThreads[n]);
	FreeMySqlData(&g_MySqlThreads[n]);
	return 0;
}

LUA_FUNCTION( thread_query_abort )
{
	int nNumParameters = g_Lua->Top();
	if (nNumParameters < 1)
	{
		g_Lua->LuaError("Insufficient parameters!\n");
		return 0;
	}

	unsigned int n = (unsigned int)g_Lua->GetInteger(1);
	n--;

	if (n >= MAX_MYSQL_THREADS || !g_MySqlThreads[n].bUsed)
	{
		g_Lua->LuaError("Invalid thread!\n");
		return 0;
	}

	g_MySqlThreads[n].bAbort = true;
	return 0;
}

LUA_FUNCTION( thread_query_complete )
{
	int nNumParameters = g_Lua->Top();
	if (nNumParameters < 1)
	{
		g_Lua->LuaError("Insufficient parameters!\n");
		return 0;
	}

	unsigned int n = (unsigned int)g_Lua->GetInteger(1);
	n--;

	if (n >= MAX_MYSQL_THREADS || !g_MySqlThreads[n].bUsed)
	{
		g_Lua->LuaError("Invalid thread!\n");
		return 0;
	}

	g_Lua->Push( g_MySqlThreads[n].bComplete );
	return 1;
}

LUA_FUNCTION( thread_query_retrieve )
{
	int nNumParameters = g_Lua->Top();
	if (nNumParameters < 1)
	{
		g_Lua->LuaError("Insufficient parameters!\n");
		return 0;
	}
	unsigned long nTimeout = INFINITE;
	if (nNumParameters > 1)
	{
		nTimeout = (unsigned long)g_Lua->GetInteger(2);
	}

	unsigned int n = (unsigned int)g_Lua->GetInteger(1);
	n--;

	if (n >= MAX_MYSQL_THREADS || !g_MySqlThreads[n].bUsed)
	{
		g_Lua->Push(false);
		g_Lua->Push(false);
		g_Lua->Push("Invalid thread!");
		return 3;
	}
	else
	{
		if (!g_MySqlThreads[n].bComplete)
		{
			WaitForMySqlThread( &g_MySqlThreads[n], nTimeout );
		}

		PushMySqlData( &g_MySqlThreads[n] );

		if (!g_MySqlThreads[n].bComplete)
		{
			ContinueMySqlThread( &g_MySqlThreads[n] );
		}
		return 3;
	}
}

#ifdef WIN32
DWORD WINAPI MySqlDataThread(void* p)
{
	DoQuery( (mysql_thread_data*)p );
	return 0;
}
#endif

LUA_FUNCTION( query )
{
	int nNumParameters = g_Lua->Top();

	if (nNumParameters < 2)
	{
		g_Lua->LuaError("Insufficient parameters!\n");
		return 0;
	}
	int nResultType = GM_MYSQL_QUERY_NUMERIC;
	if (nNumParameters > 2)
		nResultType = g_Lua->GetInteger(3);

	if ((nResultType & GM_MYSQL_QUERY_BOTH) == 0)
		nResultType |= GM_MYSQL_QUERY_NUMERIC;

	if ((nResultType & GM_MYSQL_QUERY_FLAG_MASK) == 0)
	{
		g_Lua->Push(false);
		g_Lua->Push(false);
		g_Lua->Push("Invalid result type");
		return 3;
	}

	unsigned int n = (unsigned int)g_Lua->GetInteger(1);
	n--;

	if (n >= MAX_MYSQL_CONNECTIONS || g_pMysqlInstances[n] == 0)
	{
		g_Lua->Push(false);
		g_Lua->Push(false);
		g_Lua->Push("Invalid connection!");
		return 3;
	}

	const char* sQuery = g_Lua->GetString(2);

	mysql_thread_data temp;
	memset(&temp, 0, sizeof(temp));
	temp.nResultType = nResultType;
	temp.pSQL = g_pMysqlInstances[n];
	temp.bAbort = false;
	temp.bUsed = true;
	_snprintf(temp.sQuery, sizeof(temp.sQuery), "%s", sQuery);

	if (nResultType & GM_MYSQL_QUERY_NOBLOCK)
	{
		int nThreadIndex = -1;
		for (int i=0; i<MAX_MYSQL_THREADS; i++)
		{
			if (g_MySqlThreads[i].bUsed == false)
			{
				nThreadIndex = i;
				g_MySqlThreads[i] = temp;
#ifdef WIN32
				g_MySqlThreads[i].hWaitObject = CreateEvent(0, FALSE, TRUE, 0);
				g_MySqlThreads[i].hContinueObject = CreateEvent(0, TRUE, TRUE, 0);
				g_MySqlThreads[i].hThread = CreateThread(0, 0, &MySqlDataThread, &g_MySqlThreads[i], CREATE_SUSPENDED, &g_MySqlThreads[i].dwThreadId );
				ResumeThread(g_MySqlThreads[i].hThread);
#else
	#error LINUX!
#endif
				break;
			}
		}

		if (nThreadIndex == -1)
		{
			g_Lua->Push(false);
			g_Lua->Push(false);
			g_Lua->Push("Out of MySQL threads (limit = " S(MAX_MYSQL_THREADS) ")");
		}
		else
		{
			g_Lua->Push((float)nThreadIndex+1);
			g_Lua->Push(true);
			g_Lua->Push("Invalid connection!");
		}
		return 3;
	}
	else
	{
		DoQuery(&temp);

		PushMySqlData(&temp);

		FreeMySqlData(&temp);

		return 3;
	}
}

int Init( void )
{
	// Lets make our own table filled with functions
	g_Lua->NewGlobalTable( "mysql" );
	ILuaObject* pTable = g_Lua->GetGlobal( "mysql" );

		pTable->SetMember( "connect",		connect );
		pTable->SetMember( "disconnect",	disconnect );
		pTable->SetMember( "query",			query );
		pTable->SetMember( "escape",		escape );
		pTable->SetMember( "last_insert_id",insert_id );

		pTable->SetMember( "query_read",		thread_query_retrieve	);
		pTable->SetMember( "query_complete",	thread_query_complete	);
		pTable->SetMember( "query_abort",		thread_query_abort		);
		pTable->SetMember( "query_cleanup",		thread_query_cleanup	);

		pTable->SetMember( "QUERY_NUMERIC", (float)GM_MYSQL_QUERY_NUMERIC );
		pTable->SetMember( "QUERY_FIELDS", (float)GM_MYSQL_QUERY_FIELDS );
		pTable->SetMember( "QUERY_BOTH", (float)GM_MYSQL_QUERY_BOTH );
		pTable->SetMember( "QUERY_NOBLOCK", (float)GM_MYSQL_QUERY_NOBLOCK );

	SAFE_UNREF( pTable );

	return 0;
}

int Quit( void )
{
	for (int i=0; i<MAX_MYSQL_THREADS; i++)
	{
		if (g_MySqlThreads[i].bUsed)
		{
			g_MySqlThreads[i].bAbort = true;
			WaitForThreadDeath(&g_MySqlThreads[i]);
			FreeMySqlData(&g_MySqlThreads[i]);
		}
	}

	for (int i=0; i<MAX_MYSQL_CONNECTIONS; i++)
	{
		if (g_pMysqlInstances[i])
		{
			mysql_close( g_pMysqlInstances[i] );
			g_pMysqlInstances[i] = 0;
		}
	}

	return 0;
}
